const { PrismaClient } = require("@prisma/client");

const prisma = new PrismaClient();

async function main() {
  console.log("🌱 Seeding catalogs data...\n");

  // Sample departments
  const departments = [
    { name: "Emergency Department" },
    { name: "Intensive Care Unit" },
    { name: "Operating Room" },
    { name: "Radiology" },
    { name: "Laboratory" },
    { name: "Pharmacy" },
    { name: "Cardiology" },
    { name: "Orthopedics" },
  ];

  // Sample locations for each department
  const locationsData = {
    "Emergency Department": [
      "Triage Area",
      "Trauma Room 1",
      "Trauma Room 2",
      "Examination Room A",
      "Examination Room B",
    ],
    "Intensive Care Unit": [
      "ICU Room 101",
      "ICU Room 102",
      "ICU Room 103",
      "ICU Room 104",
    ],
    "Operating Room": ["OR 1", "OR 2", "OR 3", "OR 4", "Recovery Room"],
    Radiology: [
      "X-Ray Room 1",
      "X-Ray Room 2",
      "CT Scan Room",
      "MRI Room",
      "Ultrasound Room",
    ],
    Laboratory: [
      "Blood Lab",
      "Microbiology Lab",
      "Chemistry Lab",
      "Pathology Lab",
    ],
    Pharmacy: ["Main Pharmacy", "Dispensing Area", "Storage Room"],
    Cardiology: [
      "Cardiac Catheterization Lab",
      "Echocardiography Room",
      "Stress Test Room",
    ],
    Orthopedics: ["Orthopedic Clinic A", "Orthopedic Clinic B", "Casting Room"],
  };

  // Sample elements for each location type
  const elementsData = {
    "Triage Area": [
      "Vital Signs Monitor",
      "Blood Pressure Cuff",
      "Thermometer",
      "Pulse Oximeter",
    ],
    "Trauma Room": [
      "Defibrillator",
      "Ventilator",
      "Infusion Pump",
      "ECG Machine",
      "Emergency Cart",
    ],
    "Examination Room": [
      "Examination Table",
      "Otoscope",
      "Ophthalmoscope",
      "Stethoscope",
    ],
    "ICU Room": [
      "Ventilator",
      "Patient Monitor",
      "Infusion Pump",
      "Syringe Pump",
      "Defibrillator",
    ],
    OR: [
      "Anesthesia Machine",
      "Surgical Lights",
      "Electrosurgical Unit",
      "Patient Monitor",
      "Surgical Table",
    ],
    "Recovery Room": ["Patient Monitor", "Oxygen Supply", "Suction Unit"],
    "X-Ray Room": [
      "X-Ray Machine",
      "Digital Radiography System",
      "Lead Apron Storage",
    ],
    "CT Scan Room": ["CT Scanner", "Contrast Injector", "Patient Table"],
    "MRI Room": ["MRI Scanner", "Patient Table", "MRI Compatible Monitor"],
    "Ultrasound Room": ["Ultrasound Machine", "Patient Table"],
    "Blood Lab": ["Centrifuge", "Hematology Analyzer", "Blood Gas Analyzer"],
    "Microbiology Lab": ["Microscope", "Incubator", "Autoclave"],
    "Chemistry Lab": ["Chemistry Analyzer", "Spectrophotometer"],
    "Pathology Lab": ["Microscope", "Tissue Processor", "Embedding Station"],
    "Main Pharmacy": ["Medication Dispensing System", "Refrigerator", "Safe"],
    "Dispensing Area": ["Medication Cart", "Label Printer"],
    "Storage Room": ["Medication Refrigerator", "Controlled Substance Safe"],
    "Cardiac Catheterization Lab": [
      "Cardiac Catheterization System",
      "Fluoroscopy Unit",
      "Patient Monitor",
    ],
    "Echocardiography Room": ["Echocardiography Machine", "Patient Table"],
    "Stress Test Room": ["Treadmill", "ECG Machine", "Blood Pressure Monitor"],
    "Orthopedic Clinic": [
      "Examination Table",
      "X-Ray Viewer",
      "Casting Materials",
    ],
    "Casting Room": ["Casting Table", "X-Ray Viewer"],
  };

  try {
    // Create departments
    const createdDepartments = {};
    for (const dept of departments) {
      const existing = await prisma.department.findUnique({
        where: { name: dept.name },
      });

      if (existing) {
        console.log(
          `⚠️  Department "${dept.name}" already exists, skipping...`
        );
        createdDepartments[dept.name] = existing;
      } else {
        const created = await prisma.department.create({
          data: dept,
        });
        console.log(`✅ Created department: ${dept.name}`);
        createdDepartments[dept.name] = created;
      }
    }

    console.log("\n");

    // Create locations
    const createdLocations = {};
    for (const [deptName, locationNames] of Object.entries(locationsData)) {
      const department = createdDepartments[deptName];
      if (!department) {
        console.log(
          `⚠️  Department "${deptName}" not found, skipping locations...`
        );
        continue;
      }

      for (const locName of locationNames) {
        try {
          const existing = await prisma.location.findFirst({
            where: {
              name: locName,
              departmentId: department.id,
            },
          });

          if (existing) {
            console.log(
              `⚠️  Location "${locName}" in "${deptName}" already exists, skipping...`
            );
            createdLocations[`${deptName}-${locName}`] = existing;
          } else {
            const created = await prisma.location.create({
              data: {
                name: locName,
                departmentId: department.id,
              },
            });
            console.log(`✅ Created location: ${locName} (${deptName})`);
            createdLocations[`${deptName}-${locName}`] = created;
          }
        } catch (error) {
          console.error(
            `❌ Error creating location "${locName}":`,
            error.message
          );
        }
      }
    }

    console.log("\n");

    // Create elements
    for (const [locationKey, location] of Object.entries(createdLocations)) {
      // Find matching element data
      let elementNames = [];
      for (const [pattern, elements] of Object.entries(elementsData)) {
        if (
          location.name.includes(pattern) ||
          pattern.includes(location.name)
        ) {
          elementNames = elements;
          break;
        }
      }

      // If no exact match, try partial match
      if (elementNames.length === 0) {
        for (const [pattern, elements] of Object.entries(elementsData)) {
          if (
            location.name.toLowerCase().includes(pattern.toLowerCase()) ||
            pattern.toLowerCase().includes(location.name.toLowerCase())
          ) {
            elementNames = elements;
            break;
          }
        }
      }

      // Default elements if still no match
      if (elementNames.length === 0) {
        elementNames = ["Equipment 1", "Equipment 2", "Equipment 3"];
      }

      for (const elementName of elementNames) {
        try {
          const existing = await prisma.element.findFirst({
            where: {
              name: elementName,
              locationId: location.id,
            },
          });

          if (existing) {
            console.log(
              `⚠️  Element "${elementName}" in "${location.name}" already exists, skipping...`
            );
          } else {
            await prisma.element.create({
              data: {
                name: elementName,
                locationId: location.id,
              },
            });
            console.log(
              `✅ Created element: ${elementName} (${location.name})`
            );
          }
        } catch (error) {
          console.error(
            `❌ Error creating element "${elementName}":`,
            error.message
          );
        }
      }
    }

    console.log("\n✨ Seeding completed successfully!");
  } catch (error) {
    console.error("❌ Error seeding data:", error);
    throw error;
  }
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
